#! /usr/bin/python

import sys, urllib, re, datetime, time
from xml.dom import minidom

# set up array
fullName = {
	'laa': 'Angels',
	'hou': 'Astros',
	'oak': 'Athletics',
	'tor': 'Blue Jays',
	'atl': 'Braves',
	'mil': 'Brewers',
	'stl': 'Cardinals',
	'chc': 'Cubs',
	'tam': 'Devil Rays',
	'ari': 'Diamondbacks',
	'lad': 'Dodgers',
	'sfo': 'Giants',
	'cle': 'Indians',
	'sea': 'Mariners',
	'fla': 'Marlins',
	'nym': 'Mets',
	'was': 'Nationals',
	'bal': 'Orioles',
	'sdg': 'Padres',
	'phi': 'Phillies',
	'pit': 'Pirates',
	'tex': 'Rangers',
	'bos': 'Red Sox',
	'cin': 'Reds',
	'col': 'Rockies',
	'kan': 'Royals',
	'det': 'Tigers',
	'min': 'Twins',
	'chw': 'White Sox',
	'nyy': 'Yankees'
}

# function that converts a time to the correct time zone
# ----------------------------------------------------------------------
def convertTime(timeString):

	# convert the time to the timezone
	hour = int( timeString[:-2].split(':')[0] )
	minute = int( timeString[:-2].split(':')[1] )
	
	# convert to 24 hour time
	if "PM" == timeString[-2:]:
		hour += 12
	
	# create date object
	today = datetime.date.today()
	calculatedDate = datetime.datetime( today.year, today.month, today.day)
	calculatedDate = calculatedDate + datetime.timedelta(hours=hour)
	calculatedDate = calculatedDate + datetime.timedelta(minutes=minute)
	
	# remove timezone offset
	calculatedDate = calculatedDate + datetime.timedelta(hours=timeOffset)
	
	gameTime = calculatedDate.strftime("%I:%M")
	
	# remove leading zero
	if "0" == gameTime[0]:
		gameTime = gameTime[1:]
		
	# return the converted time
	return gameTime


# function that parses a winner from the xml
# ----------------------------------------------------------------------
def getPitcherString(pitcher):

	try:
		pitcherName = pitcher.childNodes[1].getAttribute('name').split('. ')[1]
		
		if pitcherName.strip() == '.':
			dobreak()
		
		if pitcher.hasAttribute('saves'):
			return pitcherName
			
		else:
			pitcherWins = pitcher.getAttribute('wins')
			pitcherLosses = pitcher.getAttribute('losses')
			return pitcherName + ' (' + pitcherWins + '-' + pitcherLosses + ')'
		
	except:
		return ''

	
# function that parses a completed game
# ----------------------------------------------------------------------
def completedGame(input):
	
	output = []
	
	# opponent
	teams = input.getElementsByTagName('gameteam')
	prefix = 'vs '
	if fullName[team] == teams[1].parentNode.getAttribute('name'):
		(teams[0], teams[1]) = (teams[1], teams[0])
		prefix = 'at '
	output.append(prefix + teams[1].parentNode.getAttribute('name'))

	# result
	score = teams[0].getAttribute('R') + '-' + teams[1].getAttribute('R')
	
	if int(score.split('-')[0]) < int(score.split('-')[1]):
		output.append('L')
	else:
		output.append('W')
	output.append(score)
	
	# url
	output.append( 'http://mlb.mlb.com/news/boxscore.jsp?gid=' + input.getElementsByTagName('game')[0].getAttribute('id') )
	
	# pitchers
	output.append( getPitcherString( input.getElementsByTagName('w_pitcher')[0] ) )
	output.append( getPitcherString( input.getElementsByTagName('l_pitcher')[0] ) )
	
	try:
		output.append( getPitcherString( input.getElementsByTagName('sv_pitcher')[0] ) )
	except: pass

	return output


# function that parses a game in the future
# ----------------------------------------------------------------------
def futureGame(input, relativeToToday):
	
	output = []
	
	# opponent
	teams = input.getElementsByTagName('gameteam')
	prefix = 'vs '
	if fullName[team] == teams[1].parentNode.getAttribute('name'):
		(teams[0], teams[1]) = (teams[1], teams[0])
		prefix = 'at '
	opponent = teams[1].parentNode.getAttribute('name')
	output.append(prefix + opponent)
	
	# time
	time = input.getElementsByTagName('game')[0].getAttribute('start_time')
	output.append( convertTime( time ) )
	
	# game id
	gameid = input.getElementsByTagName('game')[0].getAttribute('id')
	
	# get media from appropriate source
	media = ''
	if 'XM' == mediaType:
		try:
			url = 'http://www.xmradio.com/schedule/sport/get_mlb_schedule.jsp?team_name=' + fullName[team] + '&startDate=0'
			xmsourceArray = urllib.urlopen( url )
			xmsource = xmsourceArray.readlines()
			
			# get date
			targetDate = (datetime.date.today() + datetime.timedelta(days=relativeToToday) ).strftime("/%m/%d/%Y")
			targetDate = '/'.join( targetDate.split("/0") )
			targetDate = targetDate[1:]
			
			for i in range( len(xmsource) ):
				if 0 != xmsource[i].count(targetDate+'</td>'):
					break
			
			media = ''.join( re.split('<.*?>', xmsource[i+4]) ).strip()
		except: pass
	
	elif 'ESPNTV' == mediaType or 'ESPN' == mediaType:
		try:
			url = 'http://scores.espn.go.com/mlb/teams/schedule?team=' + team
			sourceArray = urllib.urlopen( url )
			source = sourceArray.readlines()
			
			# get month
			targetDate = (datetime.date.today() + datetime.timedelta(days=relativeToToday) ).strftime("%b. '%y")
			targetDate = ' '.join( targetDate.split('  ') )
			
			for i in range( len(source) ):
				if ( re.search(targetDate.strip(), ''.join( re.split('<.*?>', source[i] ) ).strip(), 1) ):
					break
			
			# get date
			targetDate = (datetime.date.today() + datetime.timedelta(days=relativeToToday) ).strftime("%a. %e")
			targetDate = ' '.join( targetDate.split('  ') )
			
			for i in range( i, len(source) ):
				if (targetDate.strip() == ''.join( re.split('<.*?>', source[i] ) ).strip() ):
					break
			
			media = ''.join( re.split('<.*?>', source[i+3] ) ).strip()
			media = ''.join( re.split('&nbsp;', media ) )
			
		except: pass
	
	else:
		try:
			xml = minidom.parse(urllib.urlopen( baseurl + '/master_scoreboard.xml' ))
					
			# go through nodes to find game
			for game in xml.getElementsByTagName('game'):
				if gameid == game.getAttribute('gameday'):
	
					media = game.getElementsByTagName('links')[0].getAttribute('tv_station')
					break
			
		except: pass
	
	# media
	output.append(media)

	# url
	output.append( 'http://mlb.mlb.com/news/boxscore.jsp?gid=' + gameid )
	
	# starters
	output.append( getPitcherString( input.getElementsByTagName('p_pitcher')[0] ) )
	output.append( getPitcherString( input.getElementsByTagName('p_pitcher')[1] ) )
	
	return output
	

# function that parses a game in progress
# ----------------------------------------------------------------------
def inprogressGame(input):

	output = []
	
	# opponent
	teams = input.getElementsByTagName('gameteam')
	prefix = 'vs '
	if fullName[team] == teams[1].parentNode.getAttribute('name'):
		(teams[0], teams[1]) = (teams[1], teams[0])
		prefix = 'at '
	output.append(prefix + teams[1].parentNode.getAttribute('name'))

	# game status
	score = teams[0].getAttribute('R') + '-' + teams[1].getAttribute('R')
	inning = input.getElementsByTagName('inningnum')[0]
	inning = inning.getAttribute('half') + ' ' + inning.getAttribute('inning')
	
	output.append(score)
	
	gameStatus = input.getElementsByTagName('game')[0].getAttribute('status')
	if( "PRE_GAME" == gameStatus or "IMMEDIATE_PREGAME" == gameStatus ):
		output.append("T 0")
	else:
		output.append(inning)

	# url
	output.append( 'http://mlb.mlb.com/news/boxscore.jsp?gid=' + input.getElementsByTagName('game')[0].getAttribute('id') )

	# on base and outs
	xml = minidom.parse(urllib.urlopen( baseurl + '/master_scoreboard.xml' ))
			
	# go through nodes to find game
	onbase = ''
	outs = ''
	gameid = input.getElementsByTagName('game')[0].getAttribute('id')
	
	for game in xml.getElementsByTagName('game'):
		if gameid == game.getAttribute('gameday'):
			
			onbase = game.getElementsByTagName('runners_on_base')[0].getAttribute('status')
			outs = game.getElementsByTagName('status')[0].getAttribute('o')
			break

	output.append(onbase)
	output.append(outs)

	return output


# function that parses a game that has been delayed
# ----------------------------------------------------------------------
def delayedGame(input):
		
	output = []
	
	# opponent
	teams = input.getElementsByTagName('gameteam')
	prefix = 'vs '
	if fullName[team] == teams[1].parentNode.getAttribute('name'):
		(teams[0], teams[1]) = (teams[1], teams[0])
		prefix = 'at '
	output.append(prefix + teams[1].parentNode.getAttribute('name'))
	
	# delay reason
	reason = input.getElementsByTagName('delay_reason')[0].childNodes[0].nodeValue
	
	output.append( reason )
	
	return output


# function that parses a day that is a double header
# ----------------------------------------------------------------------
def doubleHeaderGame(input, relativeToToday):
	
	output = []
	
	# get games
	gameOne = parseGame(input[0], relativeToToday)
	gameTwo = parseGame(input[1], relativeToToday)
	
	output.append(gameOne[0])
	output.append('Double Header')
	
	# game one
	output.append( gameOne[1] + " - " + gameOne[2] )
	output.append( gameOne[3] ) # game id
	output.append( gameOne[4] + " " + gameOne[5] + " " + gameOne[6])
	
	# game two
	output.append( gameTwo[1] + " - " + gameTwo[2] )
	output.append( gameTwo[3] ) # game id
	output.append( gameTwo[4] + " " + gameTwo[5] + " " + gameTwo[6])
	
	return output


# function that chooses which way to parse a game
# ----------------------------------------------------------------------
def parseGame(game, relativeToToday):

	line = []

	# delayed game
	if game.getElementsByTagName('delay_reason') != []:
		line = delayedGame(game)
	
	# in progress game
	elif game.nodeName[:2] == 'ig':
		line = inprogressGame(game)
		
	# completed game
	elif game.nodeName[:2] == 'go':
		line = completedGame(game)
	
	# future game
	elif game.nodeName[:2] == 'sg':
		line = futureGame(game, relativeToToday)
	
	# make length of totalOutputLength
	while len(line) < totalOutputLength:
		line.append('')
		
	return line

# function that gets a games information from online
# ----------------------------------------------------------------------
def retrieveGame(relativeToToday):

	global baseurl
	
	# increment date if testing
	relativeToToday += dayOffset
		
	# create date we are looking for
	targetDate = datetime.date.today() + datetime.timedelta(days=relativeToToday)
	targetDateDisplay = targetDate.strftime("%a. %e")
	
	year = targetDate.strftime("%Y")
	month = targetDate.strftime("%m")
	day = targetDate.strftime("%d")
	baseurl = 'http://gd2.mlb.com/components/game/mlb/year_' + year + '/month_' + month + '/day_' + day
	
	# 404 Not Found
	if "GameDay - 404 Not Found" == urllib.urlopen(baseurl).read():
		games = []
	
	else:
	
		# connect to internet
		try:
			urllib.urlopen( baseurl + '/scoreboard.xml' )
		except:
			print "Error: there is no network connection available"
			sys.exit(2)
			
		# 404 not found error
		if urllib.urlopen( baseurl + '/scoreboard.xml' ).read() == "GameDay - 404 Not Found":
			games = []
		
		else:
		
			# get xml source
			try:
				xml = minidom.parse(urllib.urlopen( baseurl + '/scoreboard.xml' ))
			except:
				print "Error: could not parse xml"
				sys.exit(2)
			
			# go through nodes to find game
			games = []
			for currentteam in xml.getElementsByTagName('team'):
				if fullName[team] == currentteam.getAttribute('name'):
					games.append(currentteam.parentNode)
				
	line = []
	
	# No game
	if games == []:
	
		targetDate = targetDate.strftime("%Y-%m-%d")
		
		# 2007 All-Star Game
		if( targetDate == '2007-07-10' ):
			line = ['', 'All<br/>Star<br/>Game']
		# 2007 Home Run Derby
		elif( targetDate == '2007-07-09' ):
			line = ['', 'Home<br/>Run<br/>Derby']

		else:
			line = ['', 'None']
			
	# double header
	elif len(games) > 1:
		line = doubleHeaderGame(games, relativeToToday)
			
	else:
		line = parseGame(games[0], relativeToToday)
	
	# make length of totalOutputLength
	while len(line) < totalOutputLength:
		line.append('')
		
	# add date
	line.insert(0, targetDateDisplay)

	return line


# main function
# ----------------------------------------------------------------------

# quit if not correct arguments
if 7 != len(sys.argv):
	print "Error: there are not the correct number of arguments"
	sys.exit(2)

# get variables
team = sys.argv[1].lower()
dayOffset = int( sys.argv[2] )
completedGames = int( sys.argv[3] )
futureGames = int( sys.argv[4] )
mediaType = sys.argv[5]
timeOffset = int( sys.argv[6] )
totalOutputLength = 10

baseurl = ''

# loop through games
overallArray = []

for i in range(-completedGames,futureGames+1):
	for data in retrieveGame(i):
		overallArray.append( data )

# return final result
printedFirst = False

for object in overallArray:
	if not printedFirst:
		print object,
		printedFirst = True
	else:
		print ",", object,